"use client";

import { useRef, useMemo } from "react";
import { motion, useInView } from "framer-motion";
import Image from "next/image";
import { ItemWithCheck } from "@/components";
import data from "@/data/home.json";
import { useLocale } from "next-intl";

// Shared easing function
const EASE_OUT_EXPO = [0.22, 1, 0.36, 1] as const;

// Animation variants with shared easing
const variants = {
  container: {
    hidden: { opacity: 0 },
    visible: {
      opacity: 1,
      transition: { staggerChildren: 0.15 },
    },
  },
  item: {
    hidden: { opacity: 0, x: -20 },
    visible: {
      opacity: 1,
      x: 0,
      transition: { duration: 0.5, ease: EASE_OUT_EXPO },
    },
  },
  centerContent: {
    hidden: { opacity: 0, y: 20 },
    visible: {
      opacity: 1,
      y: 0,
      transition: { duration: 0.6, ease: EASE_OUT_EXPO },
    },
  },
};

// Achievement item component
const AchievementItem = ({
  title,
  description,
}: {
  title: string;
  description: string;
}) => (
  <motion.div variants={variants.item} className="flex flex-col gap-3">
    <ItemWithCheck title={title} />
    <p className="bodyText text-thirdText lg:max-w-md xl:max-w-xl">
      {description}
    </p>
  </motion.div>
);

export const CompanyAchievements = () => {
  const locale = useLocale();
  const sectionRef = useRef(null);
  const isInView = useInView(sectionRef, { once: true, margin: "-100px" });

  const achievements = data?.about?.achievements;

  // Memoize locale-dependent data
  const { highlights, details, title, subtitle, image } = useMemo(
    () => ({
      highlights:
        locale === "ar"
          ? achievements?.highlights_ar
          : achievements?.highlights_en,
      details:
        locale === "ar" ? achievements?.details_ar : achievements?.details_en,
      title: locale === "ar" ? achievements?.title_ar : achievements?.title_en,
      subtitle:
        locale === "ar" ? achievements?.subtitle_ar : achievements?.subtitle_en,
      image: achievements?.image,
    }),
    [locale, achievements]
  );

  return (
    <motion.section
      ref={sectionRef}
      initial="hidden"
      animate={isInView ? "visible" : "hidden"}
      variants={variants.container}
      className="custom-container flex flex-col lg:flex-row items-center justify-between gap-10 md:gap-5 py-10 md:py-20 xl:py-28"
    >
      {/* Left Side - Image & Info Card */}
      <motion.div
        variants={variants.centerContent}
        className="w-full lg:w-[50%] flex max-sm:flex-col items-center gap-4"
      >
        <div className="w-full sm:w-auto">
          <Image
            src={image}
            alt={title}
            width={300}
            height={385}
            className="object-cover rounded-3xl w-full sm:w-[300px] h-[385px]"
            quality={100}
          />
        </div>

        <div className="w-full sm:flex-1 lg:w-auto flex flex-col items-center justify-center lg:items-start gap-6 md:gap-12 p-6 md:p-8 rounded-2xl bg-[#EBF8FF]">
          <h2 className="h3Text text-secondary text-center lg:text-start lg:max-w-[11ch]">
            {title}
          </h2>

          <Image
            src="/images/home/star.svg"
            alt="Achievement star"
            width={106}
            height={90}
            className="self-center"
          />

          <p className="bodyText text-primary text-center lg:text-start lg:max-w-[20ch]">
            {subtitle}
          </p>
        </div>
      </motion.div>

      {/* Right Side - Achievement List */}
      <motion.div
        variants={variants.container}
        className="w-full lg:w-[45%] flex flex-col gap-6"
      >
        {highlights?.map((highlight: string, index: number) => (
          <AchievementItem
            key={`achievement-${index}`}
            title={highlight}
            description={details[index]}
          />
        ))}
      </motion.div>
    </motion.section>
  );
};
