"use client";

import { useMemo, useRef } from "react";
import Image from "next/image";
import { motion, useInView } from "framer-motion";
import { useLocale } from "next-intl";
import { CheckCircle, CheckCircle2 } from "lucide-react";
import data from "@/data/home.json";
import { CountryCard } from "@/components/country-card";

const qualityBadge = "/images/home/quality.png";

const countryAssets: Record<string, { image: string; logo: string }> = {
  Barbados: {
    image: "/images/home/barbados-img.jpg",
    logo: "/images/home/barbados.png",
  },
  // Palau: {
  //   image: "/images/home/palau-bg.jpg",
  //   logo: "/images/home/palau.png",
  // },
  "Saint Vincent and the Grenadines": {
    image: "/images/home/saint-vincent-img.jpg",
    logo: "/images/home/saint-vincent.png",
  },
};

export const FleetApprovedClassesAndFlags = () => {
  const locale = useLocale();
  const sectionRef = useRef(null);
  const isInView = useInView(sectionRef, { once: true, margin: "-160px" });

  const classesData = data?.classesAndFlags;

  const title =
    locale === "ar" ? classesData?.title_ar ?? "" : classesData?.title_en ?? "";
  const subtitle =
    locale === "ar"
      ? classesData?.subtitle_ar ?? ""
      : classesData?.subtitle_en ?? "";

  const approvedClasses =
    locale === "ar"
      ? classesData?.approvedClasses_ar ?? []
      : classesData?.approvedClasses_en ?? [];

  const flagNames =
    locale === "ar" ? classesData?.flags_ar ?? [] : classesData?.flags_en ?? [];

  const countries = useMemo(() => {
    const englishNames = classesData?.flags_en ?? [];
    const fallbackImages = classesData?.flags ?? [];

    return flagNames.map((name, index) => {
      const englishKey = englishNames[index] ?? "";
      const asset = countryAssets[englishKey] ?? undefined;
      const fallback = fallbackImages[index] ?? "/images/home/get-in-touch.jpg";

      return {
        name,
        image: asset?.image ?? fallback,
        logo: asset?.logo ?? fallback,
      };
    });
  }, [flagNames, classesData?.flags_en, classesData?.flags]);

  const classesLabel =
    locale === "ar" ? "الهيئات المعتمدة" : "Approved Classes";
  const flagsLabel = locale === "ar" ? "الأعلام" : "Flags";

  return (
    <motion.section
      ref={sectionRef}
      className="flex flex-col gap-6 md:gap-12 py-14 md:py-20 bg-[#F9F9FB] mx-6 px-5 md:px-8"
      initial={{ opacity: 0, y: 60 }}
      animate={isInView ? { opacity: 1, y: 0 } : {}}
      transition={{ duration: 0.7, ease: "easeOut" }}
    >
      <div className="mx-auto max-w-3xl text-center">
        <motion.h3
          className="h3Text text-secondary"
          initial={{ opacity: 0, y: 16 }}
          animate={isInView ? { opacity: 1, y: 0 } : {}}
          transition={{ duration: 0.6, delay: 0.1 }}
        >
          {title}
        </motion.h3>
        <motion.p
          className="bodyText text-primaryText mt-3 max-w-[45ch]"
          initial={{ opacity: 0, y: 16 }}
          animate={isInView ? { opacity: 1, y: 0 } : {}}
          transition={{ duration: 0.6, delay: 0.2 }}
        >
          {subtitle}
        </motion.p>
      </div>

      <div className="grid gap-5 md:gap-10 md:grid-cols-[minmax(0,1fr)_minmax(0,1fr)] items-center">
        <motion.div
          className="relative overflow-hidden flex items-end justify-between rounded-3xl bg-[#E8F2F6] p-8 md:p-10"
          initial={{ opacity: 0, x: -40 }}
          animate={isInView ? { opacity: 1, x: 0 } : {}}
          transition={{ duration: 0.6, delay: 0.2 }}
        >
          <div className="max-w-xl">
            <p className="h4Text text-secondary">{classesLabel}</p>
            <motion.ul
              className="mt-6 md:mt-12 flex flex-col gap-4"
              initial="hidden"
              animate={isInView ? "visible" : "hidden"}
              variants={{
                hidden: {},
                visible: {
                  transition: {
                    staggerChildren: 0.12,
                  },
                },
              }}
            >
              {approvedClasses?.map((item) => (
                <motion.li
                  key={item}
                  className="flex items-center gap-3 rounded-2xl"
                  variants={{
                    hidden: { opacity: 0, y: 20 },
                    visible: { opacity: 1, y: 0 },
                  }}
                >
                  <CheckCircle className="size-5 text-primaryText" />

                  <span className="h6Text text-primaryText">{item}</span>
                </motion.li>
              ))}
            </motion.ul>
          </div>

          <Image
            src={qualityBadge}
            alt="quality badge"
            width={140}
            height={140}
            className="w-28 sm:w-32"
          />
        </motion.div>

        <motion.div
          className="py-5 md:py-10 md:mt-4"
          initial={{ opacity: 0, x: 40 }}
          animate={isInView ? { opacity: 1, x: 0 } : {}}
          transition={{ duration: 0.6, delay: 0.3 }}
        >
          <p className="h4Text text-secondary max-md:text-center">
            {flagsLabel}
          </p>
          <div className="flex gap-4 md:gap-8 items-start justify-center mt-5 md:mt-9 px-6">
            {countries?.map((country, index) => (
              <CountryCard
                key={`${country?.name}-${index}`}
                logo={country?.logo}
                image={country?.image}
                name={country?.name}
                link="#"
                index={index}
                isVisible={isInView}
              />
            ))}
          </div>
        </motion.div>
      </div>
    </motion.section>
  );
};
