"use client";

import { useMemo, useRef } from "react";
import { motion, useInView } from "framer-motion";
import { Clock, Phone, Mail, MapPin } from "lucide-react";
import Image from "next/image";
import { useLocale } from "next-intl";
import data from "@/data/home.json";

const EASE_OUT_EXPO = [0.16, 1, 0.3, 1] as const;

const containerVariants = {
  hidden: {},
  visible: {
    transition: {
      staggerChildren: 0.2,
    },
  },
};

const leftVariants = {
  hidden: { opacity: 0, x: -50, scale: 0.95 },
  visible: {
    opacity: 1,
    x: 0,
    scale: 1,
    transition: { duration: 0.8, ease: EASE_OUT_EXPO },
  },
};

const rightVariants = {
  hidden: { opacity: 0, x: 50, scale: 0.95 },
  visible: {
    opacity: 1,
    x: 0,
    scale: 1,
    transition: { duration: 0.8, ease: EASE_OUT_EXPO },
  },
};

const hoursBoxVariants = {
  hidden: { opacity: 0, y: 30, scale: 0.9 },
  visible: {
    opacity: 1,
    y: 0,
    scale: 1,
    transition: { duration: 0.6, ease: EASE_OUT_EXPO, delay: 0.4 },
  },
};

const headerVariants = {
  hidden: { opacity: 0, y: 15, filter: "blur(6px)" },
  visible: {
    opacity: 1,
    y: 0,
    filter: "blur(0px)",
    transition: { duration: 0.5, ease: EASE_OUT_EXPO },
  },
};

const branchContainerVariants = {
  hidden: {},
  visible: {
    transition: {
      staggerChildren: 0.12,
      delayChildren: 0.2,
    },
  },
};

const branchVariants = {
  hidden: { opacity: 0, y: 25, scale: 0.97 },
  visible: {
    opacity: 1,
    y: 0,
    scale: 1,
    transition: { duration: 0.5, ease: EASE_OUT_EXPO },
  },
};

const contactItemVariants = {
  hidden: { opacity: 0, x: -10 },
  visible: {
    opacity: 1,
    x: 0,
    transition: { duration: 0.4, ease: EASE_OUT_EXPO },
  },
};

export const GetInTouch = () => {
  const locale = useLocale();
  const sectionRef = useRef<HTMLElement | null>(null);
  const isInView = useInView(sectionRef, { once: true, margin: "-100px" });

  const workingHours = data?.workingHours;
  const contact = data?.contact;

  const title =
    locale === "ar" ? contact?.title_ar ?? "" : contact?.title_en ?? "";
  const subtitle =
    locale === "ar" ? contact?.subtitle_ar ?? "" : contact?.subtitle_en ?? "";
  const hoursTitle =
    locale === "ar"
      ? workingHours?.title_ar ?? ""
      : workingHours?.title_en ?? "";
  const hoursValue =
    locale === "ar"
      ? workingHours?.hours_ar ?? ""
      : workingHours?.hours_en ?? "";

  const branches = useMemo(() => {
    return (contact?.branches ?? []).map((branch) => ({
      title: locale === "ar" ? branch?.name_ar : branch?.name_en,
      phone: branch?.phone,
      email: branch?.email,
      address: locale === "ar" ? branch?.address_ar : branch?.address_en,
    }));
  }, [contact?.branches, locale]);

  return (
    <motion.section
      ref={sectionRef}
      initial="hidden"
      animate={isInView ? "visible" : "hidden"}
      variants={containerVariants}
      className="custom-container grid md:grid-cols-2 gap-10 md:gap-16 py-10 md:py-20"
    >
      {/* Left Section - Working Hours */}
      <motion.div
        variants={leftVariants}
        className="relative rounded-3xl overflow-hidden min-h-96 h-full"
      >
        <Image
          src="/images/home/get-in-touch.jpg"
          alt="Office workspace"
          fill
          className="object-cover"
        />

        {/* Working Hours Box */}
        <motion.div
          variants={hoursBoxVariants}
          className="absolute bottom-6 left-6 right-6 bg-[#FFFFFFCC] backdrop-blur-sm rounded-2xl p-6 space-y-4"
        >
          {hoursTitle ? (
            <h4 className="h4Text text-primary">{hoursTitle}</h4>
          ) : null}

          {hoursValue ? (
            <div className="flex items-center gap-3">
              <Clock className="size-5 text-primary shrink-0" />
              <p className="h6Text text-primary">{hoursValue}</p>
            </div>
          ) : null}
        </motion.div>
      </motion.div>

      {/* Right Section - Contact Info */}
      <motion.div
        variants={rightVariants}
        className="bg-primary rounded-3xl p-6 md:p-8 lg:p-10 flex flex-col gap-8"
      >
        {/* Header */}
        <motion.div variants={headerVariants} className="space-y-3">
          {title ? <h3 className="h3Text text-secondary">{title}</h3> : null}
          {subtitle ? <p className="h6Text text-white">{subtitle}</p> : null}
        </motion.div>

        <motion.div
          variants={branchContainerVariants}
          initial="hidden"
          animate={isInView ? "visible" : "hidden"}
        >
          {branches?.map((branch, index) => (
            <motion.div
              key={`${branch?.title}-${index}`}
              variants={branchVariants}
              className="space-y-4 mb-6 last:mb-0"
            >
              <h3 className="h5Text text-white">{branch?.title}</h3>

              <motion.div
                variants={branchContainerVariants}
                className="space-y-3"
              >
                {branch?.phone ? (
                  <motion.div
                    variants={contactItemVariants}
                    className="flex items-center gap-3 group"
                  >
                    <Phone className="size-5 text-[#E7F4F9] shrink-0 group-hover:scale-110 transition-transform duration-300" />
                    <p className="bodyText text-[#E7F4F9]">{branch?.phone}</p>
                  </motion.div>
                ) : null}

                {branch?.email ? (
                  <motion.div
                    variants={contactItemVariants}
                    className="flex items-center gap-3 group"
                  >
                    <Mail className="size-5 text-[#E7F4F9] shrink-0 group-hover:scale-110 transition-transform duration-300" />
                    <p className="bodyText text-[#E7F4F9]">{branch?.email}</p>
                  </motion.div>
                ) : null}

                {branch?.address ? (
                  <motion.div variants={contactItemVariants}>
                    <a
                      href="https://maps.app.goo.gl/KWrX2A9tWGKmzAnq8?g_st=iw"
                      className="flex items-start gap-3 group"
                      target="_blank"
                    >
                      <MapPin className="size-5 text-[#E7F4F9] shrink-0 mt-0.5 group-hover:scale-110 transition-transform duration-300" />
                      <p className="bodyText text-[#E7F4F9]">
                        {branch?.address}
                      </p>
                    </a>
                  </motion.div>
                ) : null}
              </motion.div>
            </motion.div>
          ))}
        </motion.div>
      </motion.div>
    </motion.section>
  );
};
