"use client";

import Image from "next/image";
import { useRef } from "react";
import { motion, useInView } from "framer-motion";
import { cn } from "@/utils";
import { Divider } from "@heroui/react";
import data from "@/data/home.json";
import { useLocale } from "next-intl";

export type Vessel = {
  name: string;
  imo: string;
};

const vesselVariants = {
  hidden: { opacity: 0, y: 20 },
  visible: { opacity: 1, y: 0 },
};

const vesselsContainerVariants = {
  hidden: {},
  visible: {
    transition: {
      staggerChildren: 0.08,
    },
  },
};

const textVariants = {
  hidden: { opacity: 0, x: -30 },
  visible: {
    opacity: 1,
    x: 0,
    transition: { duration: 0.6 },
  },
};

const VesselCard = ({ name, imo }: { name: string; imo: string }) => (
  <motion.div
    variants={vesselVariants}
    className="flex items-center gap-3 rounded-2xl bg-[#EBF8FF] px-4 py-4"
  >
    <div className="flex items-center justify-center rounded-xl text-white p-2">
      <Image
        src="/images/fleet/hero_icon.png"
        alt="ship icon"
        width={36}
        height={36}
      />
    </div>
    <div className="flex flex-col">
      <span className="h6Text text-primaryText">{name}</span>
      <span className="smallText text-thirdText">IMO {imo}</span>
    </div>
  </motion.div>
);

const EASE_OUT_EXPO = [0.22, 1, 0.36, 1] as const;

const dividerVariants = {
  hidden: {
    scaleX: 0,
  },
  visible: {
    scaleX: 1,
    transition: {
      duration: 0.5,
      ease: EASE_OUT_EXPO,
      delay: 0.4,
    },
  },
};

export const FleetOverviewAndVessels = () => {
  const sectionRef = useRef<HTMLElement | null>(null);
  const locale = useLocale();
  const isInView = useInView(sectionRef, { once: true, margin: "-100px" });

  const fleet = data?.fleet;

  return (
    <section
      ref={sectionRef}
      className="custom-container flex items-center justify-between gap-6 md:gap-14 max-md:flex-col py-10 md:py-20 xl:py-28"
    >
      {/* Left text block */}
      <motion.div
        variants={textVariants}
        initial="hidden"
        animate={isInView ? "visible" : "hidden"}
        className="w-full md:w-[40%] max-w-full flex flex-col gap-4 md:gap-8"
      >
        <h2 className={cn("h3Text text-secondary md:max-w-[30ch]")}>
          {locale === "ar" ? fleet?.title_ar : fleet?.title_en}
        </h2>

        {/* Divider with animation */}
        <motion.div variants={dividerVariants} className="origin-left">
          <Divider className="bg-border-clr1" />
        </motion.div>

        <p
          className={cn(
            "h5Text text-primaryText text-start whitespace-pre-line"
          )}
        >
          {locale === "ar" ? fleet?.subtitle_ar : fleet?.subtitle_en}
        </p>

        <p className={cn("bodyText text-primaryText text-start")}>
          {locale === "ar" ? fleet?.description_ar : fleet?.description_en}
        </p>
      </motion.div>

      {/* Right grid with staggered cards */}
      <motion.div
        className="w-full md:w-1/2 grid min-[400px]:grid-cols-2 gap-4"
        variants={vesselsContainerVariants}
        initial="hidden"
        animate={isInView ? "visible" : "hidden"}
      >
        {fleet?.vessels?.map((vessel) => (
          <VesselCard
            key={vessel.imo}
            name={locale === "ar" ? vessel?.name_ar : vessel?.name_en}
            imo={vessel?.imo}
          />
        ))}
      </motion.div>
    </section>
  );
};
