"use client";

import { useRef } from "react";
import { motion, useInView } from "framer-motion";
import { ArrowUpRight } from "lucide-react";

const EASE_OUT_EXPO = [0.22, 1, 0.36, 1] as const;

const headerVariants = {
  hidden: { opacity: 0, y: 20 },
  visible: {
    opacity: 1,
    y: 0,
    transition: {
      duration: 0.7,
      ease: EASE_OUT_EXPO,
    },
  },
};

const containerVariants = {
  hidden: {},
  visible: {
    transition: {
      staggerChildren: 0.1,
      delayChildren: 0.15,
    },
  },
};

const cardVariants = {
  hidden: {
    opacity: 0,
    y: 30,
    scale: 0.95,
  },
  visible: {
    opacity: 1,
    y: 0,
    scale: 1,
    transition: {
      duration: 0.6,
      ease: EASE_OUT_EXPO,
    },
  },
};

type Benefit = {
  text_en: string;
  text_ar: string;
};

type KeyBenefitsProps = {
  title: string;
  benefits: Benefit[];
  locale: string;
};

export const KeyBenefits = ({ title, benefits, locale }: KeyBenefitsProps) => {
  const sectionRef = useRef<HTMLElement | null>(null);
  const isInView = useInView(sectionRef, {
    once: true,
    margin: "-100px",
    amount: 0.2,
  });

  return (
    <section ref={sectionRef} className="custom-container py-10 md:py-20">
      <div className="bg-primary rounded-2xl p-6 md:p-9">
        {/* Title */}
        <motion.h2
          variants={headerVariants}
          initial="hidden"
          animate={isInView ? "visible" : "hidden"}
          className="h3Text text-secondary text-center mb-10"
        >
          {title}
        </motion.h2>

        {/* Benefits Grid */}
        <motion.div
          variants={containerVariants}
          initial="hidden"
          animate={isInView ? "visible" : "hidden"}
          className="grid sm:grid-cols-2 lg:grid-cols-4 gap-4 md:gap-6"
        >
          {benefits?.map((benefit, index) => (
            <motion.div
              key={index}
              variants={cardVariants}
              className="bg-[#00000033] px-4.5 py-8 rounded-xl flex flex-col gap-6 group hover:bg-[#00000044] transition-colors duration-300 will-change-transform"
            >
              {/* Icon Circle */}
              <div className="size-11 rounded-full bg-secondary flex items-center justify-center shrink-0 group-hover:scale-110 transition-transform duration-300 ease-out will-change-transform">
                <ArrowUpRight className="size-5 text-white" />
              </div>

              {/* Text */}
              <p className="text-white leading-relaxed max-sm:text-sm text-base">
                {locale === "ar" ? benefit?.text_ar : benefit?.text_en}
              </p>
            </motion.div>
          ))}
        </motion.div>
      </div>
    </section>
  );
};
